#include <iostream>
#include <fstream>
#include <limits>
#include <cstdio>
#include <cmath>

#include "fft.h"
using namespace std;

const bool smooth = true;
const int nHalfWindow = 500;
const double rCutThreshold=.3;
const double epsilon = .0001;

int **gFFTBitTable = NULL;
const int MaxFastBits = 18;	// old value 16

static int IsPowerOfTwo(int x);
static int NumberOfBitsNeeded(int PowerOfTwo);
static int ReverseBits(int index, int NumBits);
static void InitFFT();

#if defined(_MSC_VER) && _MSC_VER <= 1200 // vc6
#define min(x,y) ((x)<(y)?(x):(y))
#define max(x,y) ((x)>(y)?(x):(y))
#endif

int IsPowerOfTwo(int x) {
	if (x < 2) return false;
	if (x & (x - 1)) return false;
	return true;
}

int NumberOfBitsNeeded(int PowerOfTwo) {
	int i;

	if (PowerOfTwo < 2) {
		throw("Error: FFT called with size %d\n");
	}

	for (i = 0;; i++) if (PowerOfTwo & (1 << i)) return i;
}

int ReverseBits(int index, int NumBits) {
	int i, rev;

	for (i = rev = 0; i < NumBits; i++) {
		rev = (rev << 1) | (index & 1);
		index >>= 1;
	}
	return rev;
}

void InitFFT() {
	gFFTBitTable = new int *[MaxFastBits];

	int len = 2;
	for (int b = 1; b <= MaxFastBits; b++) {
		gFFTBitTable[b - 1] = new int[len];
		for (int i = 0; i < len; i++)
			gFFTBitTable[b - 1][i] = ReverseBits(i, b);
		len <<= 1;
	}
}

void DeinitFFT() {
	if (gFFTBitTable) {
		for (int b = 1; b <= MaxFastBits; b++) {
			delete[] gFFTBitTable[b-1];
		}
		delete[] gFFTBitTable;
	}
}

inline int FastReverseBits(int i, int NumBits) {
	if (NumBits <= MaxFastBits) return gFFTBitTable[NumBits - 1][i];
	else return ReverseBits(i, NumBits);
}

/*
 * Complex Fast Fourier Transform
 * InverseTransform=true, Fourier[]/Sqrt[n]
 * InverseTransform=false, InverseFourier[]*Sqrt[n]
 */
void FFT(int NumSamples,
		bool InverseTransform,
		double *RealIn, double *ImagIn, double *RealOut, double *ImagOut)
{
	int NumBits;                 /* Number of bits needed to store indices */
	int i, j, k, n;
	int BlockSize, BlockEnd;

	double angle_numerator = 2.0 * M_PI;
	double tr, ti;                /* temp real, temp imaginary */

	if (!IsPowerOfTwo(NumSamples)) { throw("%d is not a power of two\n"); }
	if (!gFFTBitTable) InitFFT();
	if (!InverseTransform) angle_numerator = -angle_numerator;
	NumBits = NumberOfBitsNeeded(NumSamples);

	for (i = 0; i < NumSamples; i++) {
		j = FastReverseBits(i, NumBits);
		RealOut[j] = RealIn[i];
		ImagOut[j] = (ImagIn == NULL) ? 0.0 : ImagIn[i];
	}

	BlockEnd = 1;
	for (BlockSize = 2; BlockSize <= NumSamples; BlockSize <<= 1) {
		double delta_angle = angle_numerator / (double) BlockSize;
		double sm2 = sin(-2 * delta_angle);
		double sm1 = sin(-delta_angle);
		double cm2 = cos(-2 * delta_angle);
		double cm1 = cos(-delta_angle);
		double w = 2 * cm1;
		double ar0, ar1, ar2, ai0, ai1, ai2;

		for (i = 0; i < NumSamples; i += BlockSize) {
			ar2 = cm2; ar1 = cm1;
			ai2 = sm2; ai1 = sm1;
			for (j = i, n = 0; n < BlockEnd; j++, n++) {
				ar0 = w * ar1 - ar2;
				ar2 = ar1; ar1 = ar0;
				ai0 = w * ai1 - ai2;
				ai2 = ai1; ai1 = ai0;

				k = j + BlockEnd;
				tr = ar0 * RealOut[k] - ai0 * ImagOut[k];
				ti = ar0 * ImagOut[k] + ai0 * RealOut[k];

				RealOut[k] = RealOut[j] - tr;
				ImagOut[k] = ImagOut[j] - ti;

				RealOut[j] += tr;
				ImagOut[j] += ti;
			}
		}

		BlockEnd = BlockSize;
	}

	if (InverseTransform) {
		double denom = (double) NumSamples;
		for (i = 0; i < NumSamples; i++) {
			RealOut[i] /= denom;
			ImagOut[i] /= denom;
		}
	}
}


// assuming len is power of 2
void correlateHalf( double *rin, int len, double *rout){
	int n;
	int len2 = len/2;
	double *ar, *ai, *br, *bi, *cr, *ci;
	ar = new double[ len];
	ai = new double[ len];
	br = new double[ len];
	bi = new double[ len];
	cr = new double[ len];
	ci = new double[ len];
	for( n=0; n<len; n++){
		ar[n] = rin[n];
		ai[n] = 0.;
		br[n] = 0.;
		bi[n] = 0.;
	}
	for( n=0; n<len2; n++) br[n] = rin[len2-n-1];
	FFT(len, true, ar, ai, cr, ci);
	FFT(len, true, br, bi, ar, ai);
	for( n=0; n<len; n++){	// multiply	 b = a*c
		br[n] = ar[n]*cr[n] - ai[n]*ci[n];
		bi[n] = ar[n]*ci[n] + ai[n]*cr[n];
	}
	FFT( len, false, br, bi, cr, ci);
	for(n=0; n<=len2; n++) rout[n]=cr[len2+n-1]*static_cast<double>(len);
	delete [] ar;
	delete [] br;
	delete [] cr;
	delete [] ai;
	delete [] bi;
	delete [] ci;
}

void genPower( double *rr, int len, 
		double *rSpectrum, int nLenOfSpectrum,
		double *rTopPeaks, int *pnPeakPos, int nNumOfPeaks
		){
	if (len & (len - 1)) throw( "fft length of seq should be power of 2");
	int len2 = len*2;
	int lenh = len/2;
	int n;
	double sum=0.;
	for( n=0; n<len; n++) sum += rr[n];
	double mean = sum/len;
	double *r1 = new double[len];
	for( n=0; n<len; n++) r1[n] = rr[len-1-n]-mean;
	double *rlist = new double[len];
	correlateHalf( r1, len, rlist);
	for( n=1; n<=lenh; n++){
		double r = rlist[n]/(len-n)/rlist[0]*len;
		if( fabs(r)<rCutThreshold) rlist[n]=0;
		else rlist[n]=r;
	}
	rlist[0] = 1.;
	double *rar, *rai, *rbr, *rbi, *rcr, *rci;
	rar = new double[ len2];
	rai = new double[ len2];
	rbr = new double[ len2];
	rbi = new double[ len2];
	rcr = new double[ len2];
	rci = new double[ len2];
	for( n=0; n<lenh; n++){
		rar[n] = static_cast<double>(len2) * rlist[n];
		rbr[n] = rlist[n];
		rai[n] = rbi[n] = 0.;
	}
	while( n<len2){ rar[n]=rbr[n]=rai[n]=rbi[n]=0.; n++;}
	FFT(len2, true, rar, rai, rcr, rci);
	FFT(len2, false, rbr, rbi, rar, rai);
	double r;
	r = rlist[0];
	double rmax=0.000001;
	for( n=0; n<len2; n++){
		rcr[ n] = rcr[n]+rar[n]-r;
		rci[ n] -= rai[n];
		rbr[ n] = sqrt( rcr[n]*rcr[n]+ rci[n]*rci[n]);
		if( rbr[n] > rmax) rmax = rbr[n];
	}	// plist of length len2, rc_ri, absplist in rbr
	for( n=0; n<len2; n++) rbr[n] /= rmax;

	// newlist rbi
	for( n=0; n<len-1; n++){
		rbi[n] = rbr[n+len-1];
		rbi[n+len+1] = rbr[n];
	}
	rbi[len-1]=rbr[len2-2];
	rbi[len] = rbr[len2-1];

	int i;
	double rWDenominator = static_cast<double>(1./(2.*nHalfWindow+1.));
		if( smooth){
	for( n=0; n<min(len2, nLenOfSpectrum); n++){	// smooth with window size 2*w+1
		double rSum = 0.0;
		for( i=-nHalfWindow; i<=nHalfWindow; i++){
			int ni = n+i+nHalfWindow*len2;
			ni = ni % len2;
			rSum += rbr[ ni];
		}
		if( n<min(len2, nLenOfSpectrum)) rSpectrum[n] = rSum*rWDenominator;
	}
		}else{

	// copy the spectrum for caller
	for( n=0; n<min(len2, nLenOfSpectrum); n++) rSpectrum[n] = rbr[n];
		}

	const int nPeakSeekBound = nLenOfSpectrum;
	for( i=0; i<nNumOfPeaks; i++){	// top 10 peaks
		int nP=0;
		double rP = 0.;
		for( n=0; n<nPeakSeekBound; n++){
			if( rbr[n]>rP){ rP=rbr[n]; nP=n;}
		}
		pnPeakPos[i] = nP; rTopPeaks[i] = rP;
		for( n=max(0, nP-100); n<min(nPeakSeekBound,nP+100); n++) rbr[n]=0.;
	}


	delete [] r1;
	delete [] rlist;
	delete [] rar;
	delete [] rai;
	delete [] rbr; 
	delete [] rbi; 
	delete [] rcr; 
	delete [] rci;
}
